﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/s3-crt/S3CrtRequest.h>
#include <aws/s3-crt/S3Crt_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace S3Crt {
namespace Model {

/**
 */
class ListBucketsRequest : public S3CrtRequest {
 public:
  AWS_S3CRT_API ListBucketsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListBuckets"; }

  AWS_S3CRT_API Aws::String SerializePayload() const override;

  AWS_S3CRT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  AWS_S3CRT_API bool HasEmbeddedError(IOStream& body, const Http::HeaderValueCollection& header) const override;

  ///@{
  /**
   * <p>Maximum number of buckets to be returned in response. When the number is more
   * than the count of buckets that are owned by an Amazon Web Services account,
   * return all the buckets in response.</p>
   */
  inline int GetMaxBuckets() const { return m_maxBuckets; }
  inline bool MaxBucketsHasBeenSet() const { return m_maxBucketsHasBeenSet; }
  inline void SetMaxBuckets(int value) {
    m_maxBucketsHasBeenSet = true;
    m_maxBuckets = value;
  }
  inline ListBucketsRequest& WithMaxBuckets(int value) {
    SetMaxBuckets(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> <code>ContinuationToken</code> indicates to Amazon S3 that the list is being
   * continued on this bucket with a token. <code>ContinuationToken</code> is
   * obfuscated and is not a real key. You can use this
   * <code>ContinuationToken</code> for pagination of the list results. </p>
   * <p>Length Constraints: Minimum length of 0. Maximum length of 1024.</p>
   * <p>Required: No.</p>  <p>If you specify the <code>bucket-region</code>,
   * <code>prefix</code>, or <code>continuation-token</code> query parameters without
   * using <code>max-buckets</code> to set the maximum number of buckets returned in
   * the response, Amazon S3 applies a default page size of 10,000 and provides a
   * continuation token if there are more buckets.</p>
   */
  inline const Aws::String& GetContinuationToken() const { return m_continuationToken; }
  inline bool ContinuationTokenHasBeenSet() const { return m_continuationTokenHasBeenSet; }
  template <typename ContinuationTokenT = Aws::String>
  void SetContinuationToken(ContinuationTokenT&& value) {
    m_continuationTokenHasBeenSet = true;
    m_continuationToken = std::forward<ContinuationTokenT>(value);
  }
  template <typename ContinuationTokenT = Aws::String>
  ListBucketsRequest& WithContinuationToken(ContinuationTokenT&& value) {
    SetContinuationToken(std::forward<ContinuationTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Limits the response to bucket names that begin with the specified bucket name
   * prefix.</p>
   */
  inline const Aws::String& GetPrefix() const { return m_prefix; }
  inline bool PrefixHasBeenSet() const { return m_prefixHasBeenSet; }
  template <typename PrefixT = Aws::String>
  void SetPrefix(PrefixT&& value) {
    m_prefixHasBeenSet = true;
    m_prefix = std::forward<PrefixT>(value);
  }
  template <typename PrefixT = Aws::String>
  ListBucketsRequest& WithPrefix(PrefixT&& value) {
    SetPrefix(std::forward<PrefixT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Limits the response to buckets that are located in the specified Amazon Web
   * Services Region. The Amazon Web Services Region must be expressed according to
   * the Amazon Web Services Region code, such as <code>us-west-2</code> for the US
   * West (Oregon) Region. For a list of the valid values for all of the Amazon Web
   * Services Regions, see <a
   * href="https://docs.aws.amazon.com/general/latest/gr/rande.html#s3_region">Regions
   * and Endpoints</a>.</p>  <p>Requests made to a Regional endpoint that is
   * different from the <code>bucket-region</code> parameter are not supported. For
   * example, if you want to limit the response to your buckets in Region
   * <code>us-west-2</code>, the request must be made to an endpoint in Region
   * <code>us-west-2</code>.</p>
   */
  inline const Aws::String& GetBucketRegion() const { return m_bucketRegion; }
  inline bool BucketRegionHasBeenSet() const { return m_bucketRegionHasBeenSet; }
  template <typename BucketRegionT = Aws::String>
  void SetBucketRegion(BucketRegionT&& value) {
    m_bucketRegionHasBeenSet = true;
    m_bucketRegion = std::forward<BucketRegionT>(value);
  }
  template <typename BucketRegionT = Aws::String>
  ListBucketsRequest& WithBucketRegion(BucketRegionT&& value) {
    SetBucketRegion(std::forward<BucketRegionT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Map<Aws::String, Aws::String>& GetCustomizedAccessLogTag() const { return m_customizedAccessLogTag; }
  inline bool CustomizedAccessLogTagHasBeenSet() const { return m_customizedAccessLogTagHasBeenSet; }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  void SetCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag = std::forward<CustomizedAccessLogTagT>(value);
  }
  template <typename CustomizedAccessLogTagT = Aws::Map<Aws::String, Aws::String>>
  ListBucketsRequest& WithCustomizedAccessLogTag(CustomizedAccessLogTagT&& value) {
    SetCustomizedAccessLogTag(std::forward<CustomizedAccessLogTagT>(value));
    return *this;
  }
  template <typename CustomizedAccessLogTagKeyT = Aws::String, typename CustomizedAccessLogTagValueT = Aws::String>
  ListBucketsRequest& AddCustomizedAccessLogTag(CustomizedAccessLogTagKeyT&& key, CustomizedAccessLogTagValueT&& value) {
    m_customizedAccessLogTagHasBeenSet = true;
    m_customizedAccessLogTag.emplace(std::forward<CustomizedAccessLogTagKeyT>(key), std::forward<CustomizedAccessLogTagValueT>(value));
    return *this;
  }
  ///@}
 private:
  int m_maxBuckets{0};

  Aws::String m_continuationToken;

  Aws::String m_prefix;

  Aws::String m_bucketRegion;

  Aws::Map<Aws::String, Aws::String> m_customizedAccessLogTag;
  bool m_maxBucketsHasBeenSet = false;
  bool m_continuationTokenHasBeenSet = false;
  bool m_prefixHasBeenSet = false;
  bool m_bucketRegionHasBeenSet = false;
  bool m_customizedAccessLogTagHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3Crt
}  // namespace Aws
